#include "stdlib.h"
#include <MDt.h>
#include <MDtExt.h>
#include "rwcommon.h"
#include "material.h"
#include "remapper.h"
#include "float.h"

ReMapper::ReMapper(void)
{
    m_numVertices = 0;
    m_vertices = NULL;
    m_faces = NULL;
    m_numAddedFaces = 0;
    m_remapperInfos = NULL;
    m_originalFaces = NULL;
    m_numInputFaces = 0;
}

ReMapper::~ReMapper(void)
{
    m_numVertices = 0;

    if (m_vertices)
    {
        free(m_vertices);
        m_vertices = NULL;
    }

    if (m_faces)
    {
        free(m_faces);
        m_faces = NULL;
    }
}

static int
#ifdef _WINDOWS
__cdecl
#endif /* _WINDOWS */
ReMapperInfo_Compare(const void *pA,const void *pB)
{
    ReMapperInfo *mapA = (ReMapperInfo *)pA;
    ReMapperInfo *mapB = (ReMapperInfo *)pB;

    if (mapA->materialIndex < mapB->materialIndex)
    {
        return (-1);
    }
    else if (mapA->materialIndex > mapB->materialIndex)
    {
        return (1);
    }

    /* can we sort on vertex positions? */
    if (mapA->vertexIndex != mapB->vertexIndex)
    {
        if (mapA->x - mapB->x < -0.0001)
        {
            return (-1);
        }
        else if (mapA->x - mapB->x > 0.0001)
        {
            return (1);
        }
        if (mapA->y - mapB->y < -0.0001)
        {
            return (-1);
        }
        else if (mapA->y - mapB->y > 0.0001)
        {
            return (1);
        }
        if (mapA->z - mapB->z < -0.0001)
        {
            return (-1);
        }
        else if (mapA->z - mapB->z > 0.0001)
        {
            return (1);
        }
    }

    /* can we sort on vertex colors */
    if (mapA->vertexIndex != mapB->vertexIndex)
    {
        if (mapA->r - mapB->r < -0.0001)
        {
            return (-1);
        }
        else if (mapA->r - mapB->r > 0.0001)
        {
            return (1);
        }
        if (mapA->g - mapB->g < -0.0001)
        {
            return (-1);
        }
        else if (mapA->g - mapB->g > 0.0001)
        {
            return (1);
        }
        if (mapA->b - mapB->b < -0.0001)
        {
            return (-1);
        }
        else if (mapA->b - mapB->b > 0.0001)
        {
            return (1);
        }
        if (mapA->a - mapB->a < -0.0001)
        {
            return (-1);
        }
        else if (mapA->a - mapB->a > 0.0001)
        {
            return (1);
        }
    }

    /* can we sort on texture UV's? */        
    {
        if (mapA->u - mapB->u < -0.0001)
        {
            return (-1);
        }
        else if (mapA->u - mapB->u > 0.0001)
        {
            return (1);
        }

        if (mapA->v - mapB->v < -0.0001)
        {
            return (-1);
        }
        else if (mapA->v - mapB->v > 0.0001)
        {
            return (1);
        }
    }

    /* can we sort on normals? */
    if (mapA->normalIndex != mapB->normalIndex)
    {
        if (mapA->nx - mapB->nx < -0.0001)
        {
            return (-1);
        }
        else if (mapA->nx - mapB->nx > 0.0001)
        {
            return (1);
        }
        if (mapA->ny - mapB->ny < -0.0001)
        {
            return (-1);
        }
        else if (mapA->ny - mapB->ny > 0.0001)
        {
            return (1);
        }
        if (mapA->nz - mapB->nz < -0.0001)
        {
            return (-1);
        }
        else if (mapA->nz - mapB->nz > 0.0001)
        {
            return (1);
        }
    }

    /* must be equal then */
    return (0);
}

void
ReMapper::InitReMapperInfos(ReMapperInfo *infos, int numInfos)
{
    for (int i=0; i<numInfos; i++)
    {
        infos[i].materialIndex  = -1;
        infos[i].faceIndex      = -1;
        infos[i].textureIndex   = -1;
        infos[i].triIndex       = -1;
        infos[i].u              = 0.0f;
        infos[i].v              = 0.0f;
        infos[i].vertexIndex    = -1;
        infos[i].x              = 0.0f;
        infos[i].y              = 0.0f;
        infos[i].z              = 0.0f;
        infos[i].r              = 0;
        infos[i].g              = 0;
        infos[i].b              = 0;
        infos[i].a              = 255;
    }
}

void 
ReMapper::AddGroup(int shapeIndex, int groupIndex, MaterialMap *materialMap, RwV3d *tVertArray,
                   bool prelight, bool limitUVs, float uvMax)
{
    int             i, j;
    int             numVerts, numNormals, numCVerts;
    int             numFaceIndices, numTexIndices, numNormalIndices;
    int             faceIndex, polyIndex;
    long            *faceIndices;
    long            *texIndices;
    long            *normalIndices;
    DtVec3f         *mayaVerts;
    DtVec3f         *mayaNormals;    
    DtRGBA          *mayaCVerts;
    float           minu, minv;
    float           uoffset, voffset;
    int             materialID, materialIndex;
    ReMapperInfo    *currentMap;
    ReMapperInfo    *firstMap;

    DtShapeGetVertices(shapeIndex, &numVerts, &mayaVerts);
    if (DtShapeIsFlatShaded(shapeIndex))
    {
        DtShapeGetPolygonNormals(shapeIndex, &numNormals, &mayaNormals);
    }
    else
    {
        DtShapeGetNormals(shapeIndex, &numNormals, &mayaNormals);
        DtFaceGetNormalIndexByShape(shapeIndex, groupIndex, &numNormalIndices, &normalIndices);
    }

    DtShapeGetVerticesColor(shapeIndex, &numCVerts, &mayaCVerts);
    if ((!mayaCVerts) || (numCVerts != numVerts))
    {
        prelight = false;
    }

    DtFaceGetIndexByShape(shapeIndex, groupIndex, &numFaceIndices, &faceIndices);
    DtFaceGetTextureIndexByGroup(shapeIndex, groupIndex, &numTexIndices, &texIndices);
    
    if ((mayaVerts == NULL) || (faceIndices == NULL))
    {
        return;
    }

    /* Remap the material group as appropriate */
    DtMtlGetID(shapeIndex, groupIndex, &materialID);
    materialIndex = materialMap[materialID].materialMap;

    currentMap = &m_remapperInfos[m_numAddedFaces * 3];

    i = 0;
    faceIndex = 0;
    polyIndex = 0;
    
    while (i < numFaceIndices)
    {
        minu        = FLT_MAX;
        minv        = FLT_MAX;
        uoffset     = 0.0f;
        voffset     = 0.0f;
        firstMap    = currentMap;

        /* copy the first triangle */
        for (j = 0; j < 3; j++)
        {
            currentMap->materialIndex = materialIndex;
            currentMap->faceIndex = faceIndex + m_numAddedFaces;
            currentMap->triIndex = j;
            
            currentMap->vertexIndex = faceIndices[i];
            currentMap->x = mayaVerts[currentMap->vertexIndex].vec[0];
            currentMap->y = mayaVerts[currentMap->vertexIndex].vec[1];
            currentMap->z = mayaVerts[currentMap->vertexIndex].vec[2];
                       
            if (tVertArray && texIndices)
            {
                currentMap->textureIndex = texIndices[i];
                currentMap->u = tVertArray[currentMap->textureIndex].x;
                currentMap->v = tVertArray[currentMap->textureIndex].y;
                if (currentMap->u < minu)
                {
                    minu = currentMap->u;
                }
                if (currentMap->v < minv)
                {
                    minv = currentMap->v;
                }
            }
            
            if (mayaNormals)
            {
                if (DtShapeIsFlatShaded(shapeIndex))
                {
                    currentMap->normalIndex = polyIndex;
                }
                else if (normalIndices)
                {
                    currentMap->normalIndex = normalIndices[i];
                }
                currentMap->nx = mayaNormals[currentMap->normalIndex].vec[0];
                currentMap->ny = mayaNormals[currentMap->normalIndex].vec[1];
                currentMap->nz = mayaNormals[currentMap->normalIndex].vec[2];
            }

            if (prelight)
            {
                currentMap->r = mayaCVerts[currentMap->vertexIndex].r;
                currentMap->g = mayaCVerts[currentMap->vertexIndex].g;
                currentMap->b = mayaCVerts[currentMap->vertexIndex].b;
                currentMap->a = mayaCVerts[currentMap->vertexIndex].a;
            }

            currentMap++;
            i++;
        }
        
        m_originalFaces[m_numAddedFaces + faceIndex] = polyIndex;

        faceIndex++;
        
        while (faceIndices[i] != -1)
        {            
            /* copy next vertex and two earlier vertices to make 
               next triangle */
            *currentMap = *(firstMap);
            currentMap->faceIndex = faceIndex + m_numAddedFaces;
            currentMap++;
            *currentMap = *(currentMap - 2);
            currentMap->faceIndex = faceIndex + m_numAddedFaces;
            currentMap->triIndex = 1;
            currentMap++;

            currentMap->materialIndex = materialIndex;
            currentMap->faceIndex = faceIndex + m_numAddedFaces;
            currentMap->triIndex = 2;

            currentMap->vertexIndex = faceIndices[i];
            currentMap->x = mayaVerts[currentMap->vertexIndex].vec[0];
            currentMap->y = mayaVerts[currentMap->vertexIndex].vec[1];
            currentMap->z = mayaVerts[currentMap->vertexIndex].vec[2];
                       
            if (tVertArray && texIndices)
            {
                currentMap->textureIndex = texIndices[i];
                currentMap->u = tVertArray[currentMap->textureIndex].x;
                currentMap->v = tVertArray[currentMap->textureIndex].y;
                if (currentMap->u < minu)
                {
                    minu = currentMap->u;
                }
                if (currentMap->v < minv)
                {
                    minv = currentMap->v;
                }
            }

            if (mayaNormals)
            {
                if (DtShapeIsFlatShaded(shapeIndex))
                {
                    currentMap->normalIndex = polyIndex;
                }
                else if (normalIndices)
                {
                    currentMap->normalIndex = normalIndices[i];
                }
                currentMap->nx = mayaNormals[currentMap->normalIndex].vec[0];
                currentMap->ny = mayaNormals[currentMap->normalIndex].vec[1];
                currentMap->nz = mayaNormals[currentMap->normalIndex].vec[2];
            }
            
            if (prelight)
            {
                currentMap->r = mayaCVerts[currentMap->vertexIndex].r;
                currentMap->g = mayaCVerts[currentMap->vertexIndex].g;
                currentMap->b = mayaCVerts[currentMap->vertexIndex].b;
                currentMap->a = mayaCVerts[currentMap->vertexIndex].a;
            }

            m_originalFaces[m_numAddedFaces + faceIndex] = polyIndex;
            
            currentMap++;
            i++;
            faceIndex++;
        } 
        
        i++;
        polyIndex++;
        
        /* keep uv's within a specified 0->x limit if required */
        if (limitUVs)
        {                    
            if (minu > uvMax)
            {
                uoffset = (float)fmod(minu, uvMax) - (float)minu;
            }
            if (minv > uvMax)
            {
                voffset = (float)fmod(minv, uvMax) - (float)minv;
            }
            while(firstMap != currentMap)
            {
                firstMap->u += uoffset;
                firstMap->v += voffset;
                firstMap++;
            }
        }
        
    }

    m_numAddedFaces += faceIndex;

    free(texIndices);
}

int
ReMapper::DoRemap()
{
    m_numFaces = m_numAddedFaces;

    /* sort on vertexIndex->colorIndex->normalIndex */
    qsort(m_remapperInfos, m_numAddedFaces * 3, sizeof(ReMapperInfo), ReMapperInfo_Compare);

    /* now remap the vertex face triangles */
    {
        int mapNum;
        int numFinalVertices = 0;
        int currentVertexIndex = -1;

        m_faces = (ReMapperFaceLoop *)malloc(sizeof(ReMapperFaceLoop) * m_numAddedFaces);
        if (!m_faces)
        {
            free(m_remapperInfos);
            return (false);
        }

        for (mapNum = 0; mapNum < m_numAddedFaces * 3; mapNum++)
        {
            int triIndex;
            int faceIndex = m_remapperInfos[mapNum].faceIndex;

            /* do we increament the vertex index? */
            if ((mapNum == 0) || 
                (ReMapperInfo_Compare(&m_remapperInfos[mapNum-1], &m_remapperInfos[mapNum]) != 0))
            {
                currentVertexIndex++;
            }

            /* remap its associated face */            			
            triIndex = m_remapperInfos[mapNum].triIndex;
            m_faces[faceIndex].index[triIndex]  = currentVertexIndex;
            m_faces[faceIndex].faceNum          = m_originalFaces[faceIndex];
            m_faces[faceIndex].materialIndex    = m_remapperInfos[mapNum].materialIndex;
        }

        numFinalVertices = currentVertexIndex + 1;
        m_vertices = (ReMapperVertex *)malloc(sizeof(ReMapperVertex) * numFinalVertices);
        m_numVertices = numFinalVertices;
        if (!m_vertices)
        {
            free(m_faces);
            free(m_remapperInfos);
            return (false);
        }

        currentVertexIndex = -1;

        for (mapNum = 0; mapNum < m_numAddedFaces * 3; mapNum++)
        {
            if ((mapNum == 0) || 
                (ReMapperInfo_Compare(&m_remapperInfos[mapNum-1], &m_remapperInfos[mapNum]) != 0))
            {
                int vertexIndex, faceIndex, normalIndex, materialIndex;

                currentVertexIndex++;

                /* get vertexIndex */
                vertexIndex = m_remapperInfos[mapNum].vertexIndex;
                m_vertices[currentVertexIndex].vertexIndex = vertexIndex;
                
                /* get texture coords */                
                m_vertices[currentVertexIndex].u = m_remapperInfos[mapNum].u;
                m_vertices[currentVertexIndex].v = m_remapperInfos[mapNum].v;

                /* get textureIndex */
                normalIndex = m_remapperInfos[mapNum].normalIndex;
                m_vertices[currentVertexIndex].normalIndex = normalIndex;

                /* get faceIndex */
                faceIndex = m_remapperInfos[mapNum].faceIndex;
                m_vertices[currentVertexIndex].faceIndex = faceIndex;

                /* get groupIndex */
                materialIndex = m_remapperInfos[mapNum].materialIndex;
                m_vertices[currentVertexIndex].materialIndex = materialIndex;

                /* get CVerts */
                m_vertices[currentVertexIndex].r = m_remapperInfos[mapNum].r;
                m_vertices[currentVertexIndex].g = m_remapperInfos[mapNum].g;
                m_vertices[currentVertexIndex].b = m_remapperInfos[mapNum].b;
                m_vertices[currentVertexIndex].a = m_remapperInfos[mapNum].a;
            }
        }
    }

    /* done its job */
    free(m_remapperInfos);
    free(m_originalFaces);

    /* all done beautifully */
    return true;
}

void
ReMapper::ResetReMap(void)
{
    m_numVertices   = 0;
    m_numAddedFaces = 0;
    m_numInputFaces = 0;

    if (m_vertices)
    {
        free(m_vertices);
        m_vertices = NULL;
    }

    if (m_faces)
    {
        free(m_faces);
        m_faces = NULL;
    }

    if (m_remapperInfos)
    {
        free(m_remapperInfos);
        m_remapperInfos = NULL;
    }
}

ReMapperFaceLoop *
ReMapper::GetFace(int faceNum)
{
    if (m_faces)
    {
        return (&m_faces[faceNum]);
    }

    return (NULL);
}

ReMapperVertex *
ReMapper::GetVertex(int vertexNum)
{
    if (m_vertices)
    {
        return (&m_vertices[vertexNum]);
    }

    return (NULL);
}

ReMapper::GetNumVertices(void)
{
    return (m_numVertices);
}

ReMapper::GetNumFaces(void)
{
    return (m_numFaces);
}

void
ReMapper::SetNumInputFaces(int numFaces)
{
    if (m_remapperInfos)
    {
        m_remapperInfos = (ReMapperInfo *)RwRealloc(m_remapperInfos, sizeof(ReMapperInfo) * numFaces * 3);
    }
    else
    {
        m_remapperInfos = (ReMapperInfo *)RwMalloc(sizeof(ReMapperInfo) * numFaces * 3);
    }

    InitReMapperInfos(&m_remapperInfos[m_numAddedFaces*3], (numFaces-m_numAddedFaces)*3);    

    if (m_originalFaces)
    {
        m_originalFaces = (int *)RwRealloc(m_originalFaces, sizeof(int) * numFaces);
    }
    else
    {
        m_originalFaces = (int *)RwMalloc(sizeof(int) * numFaces);
    }

    m_numInputFaces += numFaces;
}